<?php

/**
 * NetGains Affiliate – General Tracking Tag Builder
 *
 * Docs: https://affiliate.netgains.biz/manual/general-tracking
 *
 * - Build a <img> tracking pixel tag
 * - Automatically validate and fill default fields
 * - Optionally generate a secure hash (sh) for anti-tampering
 */
class Class_NetGains_Affiliate
{
    private const AFFILIATE_SERVER_HOST = 'https://affiliate.netgains.biz';
    private const AFFILIATE_SERVER_PATH = '/tracking/general_tracking';

    /**
     * Build the tracking IMG tag.
     *
     * @param array{
     *   shop_id: string,
     *   unique_id?: string,
     *   uname?: string,
     *   email?: string,
     *   amount?: int|float|string,
     *   ts?: string,
     *   unit_num?: int|string
     * } $params
     * @param string $secret Optional password to generate secure hash (sh). If empty, skipped.
     *
     * @return string
     */
    public static function getImgTag(array $params, string $secret = ''): string
    {
        $data = self::normalizeAndValidate($params);

        if (!$data) {
            return '';
        }
        // Add secure hash if secret is provided
        if (!empty($secret)) {
            $data['sh'] = self::computeSecureHash($data, $secret);
        }

        $query = http_build_query($data);
        $src = self::AFFILIATE_SERVER_HOST . self::AFFILIATE_SERVER_PATH . '?' . $query;

        return sprintf('<img src="%s" width="1" height="1" >', $src);
    }

    /**
     * Normalize & validate input, return cleaned array ready for query.
     *
     * @param array $params
     * @return array
     */
    private static function normalizeAndValidate(array $params): array
    {
        $isOk = true;
        // shop_id: required
        if (empty($params['shop_id'])) {
            echo "ERROR !! : 'shop_id' is required.<br>\n";
            $isOk = false;
        }

        // amount: default "0" if missing; must be numeric
        if (empty($params['amount'])) {
            $params['amount'] = 0;
        } else {
            $params['amount'] = (int) $params['amount'];
        }

        // uname: if present, must match regex (max 32 chars, a-z0-9, _, -, ,)
        if (isset($param['uname']) && !preg_match('/^[a-z0-9,_-]{0,32}$/i', $params['uname'])) {
            echo "ERROR !! : invalid 'uname' value.<br>\n";
            $isOk = false;
        }

        // unit_num: default "0"; if present must be integer
        if (isset($params['unit_num']) && !is_numeric($params['unit_num']) && $params['unit_num'] != "") {
        	echo("ERROR !! : 'unit_num' must be integer.<br>\n");
            $isOk = false;
        }

        // ts: default to now (YmdHis); must be 14-digit string
        if (isset($params['ts'])) {
            if (!preg_match('/^[0-9]{14}$/', $params['ts'])) {
                echo "ERROR !! : invalid 'ts' value.<br>\n";
                $isOk = false;
            }
        } else {
            $params['ts'] = date('YmdHis');
        }

        if (!$isOk) {
            return [];
        }

        return $params;
    }

    /**
     * Compute secure hash (sha1) in original field order.
     *
     * Note: Kept legacy sha1 concatenation for compatibility.
     * If supported, prefer hash_hmac('sha1', ...) in the future.
     *
     * @param array $data
     * @param string $secret
     * @return string
     */
    private static function computeSecureHash(array $data, string $secret): string
    {
        $keys = ['shop_id', 'unique_id', 'uname', 'email', 'amount', 'ts', 'unit_num'];
        $buf = $secret;

        foreach ($keys as $k) {
            $buf .= $data[$k] ?? '';
        }

        return sha1($buf);
    }
}
